import React, { useState } from 'react';
import { BrowserRouter as Router, Routes, Route, Navigate } from 'react-router-dom';
import { motion } from 'framer-motion';
import { Scale, FileText, Search, Users, Shield, Sparkles } from 'lucide-react';
import { Button } from '@/components/ui/button.jsx';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card.jsx';
import { Badge } from '@/components/ui/badge.jsx';
import Dashboard from './components/Dashboard.jsx';
import ContractGenerator from './components/ContractGenerator.jsx';
import ContractReview from './components/ContractReview.jsx';
import ClauseDrafter from './components/ClauseDrafter.jsx';
import TemplateLibrary from './components/TemplateLibrary.jsx';
import ContractDetail from './components/ContractDetail.jsx';
import './App.css';

function LandingPage({ onGetStarted }) {
  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 via-white to-purple-50">
      {/* Header */}
      <header className="container mx-auto px-6 py-8">
        <nav className="flex items-center justify-between">
          <motion.div 
            className="flex items-center space-x-2"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.5 }}
          >
            <Scale className="h-8 w-8 text-blue-600" />
            <span className="text-2xl font-bold text-gray-900">LAWAI</span>
          </motion.div>
          
          <motion.div
            initial={{ opacity: 0, x: 20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.5, delay: 0.2 }}
          >
            <Button onClick={onGetStarted} className="bg-blue-600 hover:bg-blue-700">
              Get Started
            </Button>
          </motion.div>
        </nav>
      </header>

      {/* Hero Section */}
      <section className="container mx-auto px-6 py-16 text-center">
        <motion.div
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.7 }}
        >
          <h1 className="text-5xl md:text-6xl font-bold text-gray-900 mb-6">
            AI-Powered Legal Assistant for
            <span className="text-blue-600 block">Saudi Arabia & Jordan</span>
          </h1>
          
          <p className="text-xl text-gray-600 mb-8 max-w-3xl mx-auto">
            Draft, review, and manage contracts with confidence using our AI assistant 
            specialized in Saudi Arabian and Jordanian law. Powered by Google Gemini.
          </p>
          
          <div className="flex flex-wrap justify-center gap-4 mb-12">
            <Badge variant="secondary" className="px-4 py-2 text-sm">
              <Shield className="h-4 w-4 mr-2" />
              Sharia Law Compliant
            </Badge>
            <Badge variant="secondary" className="px-4 py-2 text-sm">
              <Sparkles className="h-4 w-4 mr-2" />
              Google Gemini Powered
            </Badge>
            <Badge variant="secondary" className="px-4 py-2 text-sm">
              <Users className="h-4 w-4 mr-2" />
              Multi-Jurisdiction Support
            </Badge>
          </div>
          
          <Button 
            onClick={onGetStarted}
            size="lg" 
            className="bg-blue-600 hover:bg-blue-700 text-lg px-8 py-4"
          >
            Start Creating Contracts
          </Button>
        </motion.div>
      </section>

      {/* Features Section */}
      <section className="container mx-auto px-6 py-16">
        <motion.div
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.7, delay: 0.3 }}
          className="text-center mb-16"
        >
          <h2 className="text-4xl font-bold text-gray-900 mb-4">
            Everything You Need for Legal Excellence
          </h2>
          <p className="text-xl text-gray-600">
            Comprehensive tools designed for legal professionals
          </p>
        </motion.div>

        <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
          {[
            {
              icon: FileText,
              title: "Contract Generation",
              description: "Generate comprehensive contracts tailored to Saudi Arabian and Jordanian law with AI assistance.",
              color: "text-blue-600"
            },
            {
              icon: Search,
              title: "Contract Review",
              description: "Analyze contracts for compliance, risks, and legal issues with detailed recommendations.",
              color: "text-green-600"
            },
            {
              icon: Scale,
              title: "Clause Drafting",
              description: "Draft specific contract clauses with jurisdiction-specific legal requirements.",
              color: "text-purple-600"
            },
            {
              icon: Users,
              title: "Template Library",
              description: "Access a comprehensive library of legal templates for various contract types.",
              color: "text-orange-600"
            },
            {
              icon: Shield,
              title: "Compliance Check",
              description: "Ensure your contracts comply with local laws and regulations automatically.",
              color: "text-red-600"
            },
            {
              icon: Sparkles,
              title: "AI Legal Advice",
              description: "Get instant legal guidance and recommendations from our AI assistant.",
              color: "text-indigo-600"
            }
          ].map((feature, index) => (
            <motion.div
              key={index}
              initial={{ opacity: 0, y: 30 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ duration: 0.5, delay: 0.1 * index }}
            >
              <Card className="h-full hover:shadow-lg transition-shadow duration-300">
                <CardHeader>
                  <feature.icon className={`h-12 w-12 ${feature.color} mb-4`} />
                  <CardTitle className="text-xl">{feature.title}</CardTitle>
                </CardHeader>
                <CardContent>
                  <CardDescription className="text-gray-600">
                    {feature.description}
                  </CardDescription>
                </CardContent>
              </Card>
            </motion.div>
          ))}
        </div>
      </section>

      {/* CTA Section */}
      <section className="bg-blue-600 text-white py-16">
        <div className="container mx-auto px-6 text-center">
          <motion.div
            initial={{ opacity: 0, y: 30 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.7 }}
          >
            <h2 className="text-4xl font-bold mb-4">
              Ready to Transform Your Legal Practice?
            </h2>
            <p className="text-xl mb-8 opacity-90">
              Join legal professionals who trust LAWAI for their contract needs
            </p>
            <Button 
              onClick={onGetStarted}
              size="lg" 
              variant="secondary"
              className="text-lg px-8 py-4"
            >
              Get Started Now
            </Button>
          </motion.div>
        </div>
      </section>

      {/* Footer */}
      <footer className="bg-gray-900 text-white py-8">
        <div className="container mx-auto px-6 text-center">
          <div className="flex items-center justify-center space-x-2 mb-4">
            <Scale className="h-6 w-6" />
            <span className="text-xl font-bold">LAWAI</span>
          </div>
          <p className="text-gray-400">
            AI-Powered Legal Assistant for Saudi Arabia & Jordan
          </p>
        </div>
      </footer>
    </div>
  );
}

function App() {
  const [showDashboard, setShowDashboard] = useState(false);

  if (!showDashboard) {
    return <LandingPage onGetStarted={() => setShowDashboard(true)} />;
  }

  return (
    <Router>
      <div className="min-h-screen bg-gray-50">
        <Routes>
          <Route path="/" element={<Dashboard />} />
          <Route path="/generate" element={<ContractGenerator />} />
          <Route path="/review" element={<ContractReview />} />
          <Route path="/clauses" element={<ClauseDrafter />} />
          <Route path="/templates" element={<TemplateLibrary />} />
          <Route path="/contracts/:id" element={<ContractDetail />} />
          <Route path="*" element={<Navigate to="/" replace />} />
        </Routes>
      </div>
    </Router>
  );
}

export default App;

