import React, { useState } from 'react';
import { Link } from 'react-router-dom';
import { motion } from 'framer-motion';
import { 
  Scale, 
  ArrowLeft, 
  FileText, 
  Sparkles,
  Download,
  Save,
  Settings
} from 'lucide-react';
import { Button } from '@/components/ui/button.jsx';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card.jsx';
import { Input } from '@/components/ui/input.jsx';
import { Label } from '@/components/ui/label.jsx';
import { Textarea } from '@/components/ui/textarea.jsx';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select.jsx';
import { Badge } from '@/components/ui/badge.jsx';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs.jsx';
import { API_BASE_URL } from '../config'; // Import the API_BASE_URL

const contractTypes = [
  'Employment Agreement',
  'Commercial Lease',
  'Service Agreement',
  'Partnership Agreement',
  'Non-Disclosure Agreement',
  'Supply Agreement',
  'Distribution Agreement',
  'Consulting Agreement'
];

const jurisdictions = [
  { value: 'SA', label: 'Saudi Arabia' },
  { value: 'JO', label: 'Jordan' }
];

function ContractGenerator() {
  const [formData, setFormData] = useState({
    title: '',
    contractType: '',
    jurisdiction: '',
    parties: {
      party1: { name: '', type: 'individual', address: '' },
      party2: { name: '', type: 'individual', address: '' }
    },
    requirements: '',
    additionalInput: ''
  });
  
  const [generatedContract, setGeneratedContract] = useState('');
  const [isGenerating, setIsGenerating] = useState(false);

  const handleInputChange = (field, value) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const handlePartyChange = (party, field, value) => {
    setFormData(prev => ({
      ...prev,
      parties: {
        ...prev.parties,
        [party]: {
          ...prev.parties[party],
          [field]: value
        }
      }
    }));
  };

  const generateContract = async () => {
    setIsGenerating(true);
    try {
      const response = await fetch(`${API_BASE_URL}/api/contracts/generate`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(formData),
      });

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const data = await response.json();
//      setGeneratedContract(data.contractText); // Assuming backend returns { contractText: "..." }
        setGeneratedContract(data.content);
    } catch (error) {
      console.error('Error generating contract:', error);
      setGeneratedContract('Error: Could not generate contract. Please check your backend connection and try again.');
    } finally {
      setIsGenerating(false);
    }
  };

  const saveContract = () => {
    // Simulate saving contract
    alert('Contract saved successfully!');
  };

  const downloadContract = () => {
    const element = document.createElement('a');
    const file = new Blob([generatedContract], { type: 'text/plain' });
    element.href = URL.createObjectURL(file);
    element.download = `${formData.title || 'contract'}.txt`;
    document.body.appendChild(element);
    element.click();
    document.body.removeChild(element);
  };

  return (
    <div className="flex">
      {/* Sidebar */}
      <div className="w-64 bg-white shadow-lg h-screen fixed left-0 top-0 z-10">
        <div className="p-6 border-b">
          <Link to="/" className="flex items-center space-x-2 text-gray-600 hover:text-gray-900">
            <ArrowLeft className="h-5 w-5" />
            <span>Back to Dashboard</span>
          </Link>
        </div>
        <div className="p-6">
          <div className="flex items-center space-x-2 mb-4">
            <Scale className="h-6 w-6 text-blue-600" />
            <span className="text-xl font-bold text-gray-900">LAWAI</span>
          </div>
          <Badge variant="secondary" className="w-full justify-center">
            Contract Generator
          </Badge>
        </div>
      </div>

      {/* Main Content */}
      <div className="flex-1 ml-64 p-8">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
        >
          <div className="mb-8">
            <h1 className="text-3xl font-bold text-gray-900 mb-2">Generate Contract</h1>
            <p className="text-gray-600">Create comprehensive legal contracts with AI assistance</p>
          </div>

          <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
            {/* Form Section */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center space-x-2">
                  <Settings className="h-5 w-5" />
                  <span>Contract Details</span>
                </CardTitle>
                <CardDescription>
                  Provide the basic information for your contract
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                <div>
                  <Label htmlFor="title">Contract Title</Label>
                  <Input
                    id="title"
                    placeholder="e.g., Software Development Agreement"
                    value={formData.title}
                    onChange={(e) => handleInputChange('title', e.target.value)}
                  />
                </div>

                <div>
                  <Label htmlFor="contractType">Contract Type</Label>
                  <Select onValueChange={(value) => handleInputChange('contractType', value)}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select contract type" />
                    </SelectTrigger>
                    <SelectContent>
                      {contractTypes.map((type) => (
                        <SelectItem key={type} value={type}>
                          {type}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <div>
                  <Label htmlFor="jurisdiction">Jurisdiction</Label>
                  <Select onValueChange={(value) => handleInputChange('jurisdiction', value)}>
                    <SelectTrigger>
                      <SelectValue placeholder="Select jurisdiction" />
                    </SelectTrigger>
                    <SelectContent>
                      {jurisdictions.map((jurisdiction) => (
                        <SelectItem key={jurisdiction.value} value={jurisdiction.value}>
                          {jurisdiction.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>

                <Tabs defaultValue="party1" className="w-full">
                  <TabsList className="grid w-full grid-cols-2">
                    <TabsTrigger value="party1">Party 1</TabsTrigger>
                    <TabsTrigger value="party2">Party 2</TabsTrigger>
                  </TabsList>
                  
                  <TabsContent value="party1" className="space-y-4">
                    <div>
                      <Label htmlFor="party1Name">Name</Label>
                      <Input
                        id="party1Name"
                        placeholder="Full name or company name"
                        value={formData.parties.party1.name}
                        onChange={(e) => handlePartyChange('party1', 'name', e.target.value)}
                      />
                    </div>
                    <div>
                      <Label htmlFor="party1Type">Type</Label>
                      <Select onValueChange={(value) => handlePartyChange('party1', 'type', value)}>
                        <SelectTrigger>
                          <SelectValue placeholder="Select type" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="individual">Individual</SelectItem>
                          <SelectItem value="company">Company</SelectItem>
                          <SelectItem value="organization">Organization</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>
                    <div>
                      <Label htmlFor="party1Address">Address</Label>
                      <Textarea
                        id="party1Address"
                        placeholder="Full address"
                        value={formData.parties.party1.address}
                        onChange={(e) => handlePartyChange('party1', 'address', e.target.value)}
                      />
                    </div>
                  </TabsContent>
                  
                  <TabsContent value="party2" className="space-y-4">
                    <div>
                      <Label htmlFor="party2Name">Name</Label>
                      <Input
                        id="party2Name"
                        placeholder="Full name or company name"
                        value={formData.parties.party2.name}
                        onChange={(e) => handlePartyChange('party2', 'name', e.target.value)}
                      />
                    </div>
                    <div>
                      <Label htmlFor="party2Type">Type</Label>
                      <Select onValueChange={(value) => handlePartyChange('party2', 'type', value)}>
                        <SelectTrigger>
                          <SelectValue placeholder="Select type" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="individual">Individual</SelectItem>
                          <SelectItem value="company">Company</SelectItem>
                          <SelectItem value="organization">Organization</SelectItem>
                        </SelectContent>
                      </Select>
                    </div>
                    <div>
                      <Label htmlFor="party2Address">Address</Label>
                      <Textarea
                        id="party2Address"
                        placeholder="Full address"
                        value={formData.parties.party2.address}
                        onChange={(e) => handlePartyChange('party2', 'address', e.target.value)}
                      />
                    </div>
                  </TabsContent>
                </Tabs>

                <div>
                  <Label htmlFor="requirements">Specific Requirements</Label>
                  <Textarea
                    id="requirements"
                    placeholder="Describe the specific terms, conditions, and requirements for this contract..."
                    rows={4}
                    value={formData.requirements}
                    onChange={(e) => handleInputChange('requirements', e.target.value)}
                  />
                </div>

                <div>
                  <Label htmlFor="additionalInput">Additional Instructions</Label>
                  <Textarea
                    id="additionalInput"
                    placeholder="Any additional instructions or special considerations..."
                    rows={3}
                    value={formData.additionalInput}
                    onChange={(e) => handleInputChange('additionalInput', e.target.value)}
                  />
                </div>

                <Button 
                  onClick={generateContract}
                  disabled={!formData.contractType || !formData.jurisdiction || isGenerating}
                  className="w-full"
                >
                  {isGenerating ? (
                    <>
                      <Sparkles className="h-4 w-4 mr-2 animate-spin" />
                      Generating Contract...
                    </>
                  ) : (
                    <>
                      <Sparkles className="h-4 w-4 mr-2" />
                      Generate Contract
                    </>
                  )}
                </Button>
              </CardContent>
            </Card>

            {/* Generated Contract Section */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center space-x-2">
                  <FileText className="h-5 w-5" />
                  <span>Generated Contract</span>
                </CardTitle>
                <CardDescription>
                  Your AI-generated contract will appear here
                </CardDescription>
              </CardHeader>
              <CardContent>
                {generatedContract ? (
                  <div className="space-y-4">
                    <div className="flex space-x-2">
                      <Button onClick={saveContract} variant="outline" size="sm">
                        <Save className="h-4 w-4 mr-2" />
                        Save
                      </Button>
                      <Button onClick={downloadContract} variant="outline" size="sm">
                        <Download className="h-4 w-4 mr-2" />
                        Download
                      </Button>
                    </div>
                    <div className="bg-gray-50 p-4 rounded-lg max-h-96 overflow-y-auto">
                      <pre className="whitespace-pre-wrap text-sm text-gray-800">
                        {generatedContract}
                      </pre>
                    </div>
                  </div>
                ) : (
                  <div className="text-center py-12 text-gray-500">
                    <FileText className="h-12 w-12 mx-auto mb-4 opacity-50" />
                    <p>Fill out the form and click "Generate Contract" to see your contract here</p>
                  </div>
                )}
              </CardContent>
            </Card>
          </div>
        </motion.div>
      </div>
    </div>
  );
}

export default ContractGenerator;

